-- RoN_HelmetCam/scripts/main.lua
local function log(s) print("[RoN_HelmetCam] " .. s .. "\n") end

local state = {
    enabled = false,
    current_mode = nil, -- "helmet" или "high"
    cam_comp = nil,
    applied = { X = 0.0, Y = 0.0, Z = 0.0 },
}

-- Настройки смещений:
local HELMET_OFFSET = { X = -10.0, Y = -6.8, Z = -7.2} -- Вниз
local HIGH_CAM_OFFSET = { X = -20.0, Y = -8.0, Z = 15.0}  -- Вверх (настройте под себя)

local function safe(obj)
    if obj == nil then return false end
    if obj.IsValid and not obj:IsValid() then return false end
    return true
end

local function try_prop(obj, prop)
    if not safe(obj) then return nil end
    local ok, v = pcall(function() return obj[prop] end)
    if ok then return v end
    return nil
end

local function get_pc()
    local engine = FindFirstOf("Engine")
    if not safe(engine) then return nil end
    local gv = engine.GameViewport
    if not safe(gv) then return nil end
    local world = gv.World
    if not safe(world) then return nil end
    local gi = world.OwningGameInstance
    if not safe(gi) then return nil end
    local locals = gi.LocalPlayers
    if not locals then return nil end
    local lp = locals[1] or locals[0]
    if not safe(lp) then return nil end
    return lp.PlayerController
end

local function get_pawn(pc)
    if not safe(pc) then return nil end
    local ok, pawn = pcall(function() return pc:GetPawn() end)
    if ok and safe(pawn) then return pawn end
    return try_prop(pc, "AcknowledgedPawn")
end

local function find_camera_component(pawn)
    if not safe(pawn) then return nil, "pawn invalid" end
    local candidates = {"FirstPersonCameraComponent", "FirstPersonCamera", "CameraComponent", "FPVCamera"}
    for _, name in ipairs(candidates) do
        local comp = try_prop(pawn, name)
        if safe(comp) then return comp, name end
    end
    return nil, "not found"
end

local function add_local_offset(comp, vec)
    if not safe(comp) then return false, "comp invalid" end
    local outHit = {} 
    local func = comp.K2_AddLocalOffset or comp.AddLocalOffset
    if func then
        local ok, err = pcall(function() func(comp, vec, false, outHit, true) end)
        if ok then return true end
        return false, tostring(err)
    end
    return false, "no offset function"
end

-- Функция отключения текущего оффсета
local function disable_offset()
    if not state.enabled or not safe(state.cam_comp) then 
        return true 
    end

    local back = { X = -state.applied.X, Y = -state.applied.Y, Z = -state.applied.Z }
    add_local_offset(state.cam_comp, back)
    
    state.enabled = false
    state.cam_comp = nil
    state.current_mode = nil
    state.applied = { X = 0.0, Y = 0.0, Z = 0.0 }
    return true
end

-- Универсальная функция включения
local function enable_offset(mode)
    -- Если уже что-то включено, сначала выключаем
    if state.enabled then disable_offset() end

    local pc = get_pc()
    local pawn = get_pawn(pc)
    local cam, how = find_camera_component(pawn)

    if not safe(cam) then return false, "Camera not found" end

    local offset_data = (mode == "helmet") and HELMET_OFFSET or HIGH_CAM_OFFSET
    
    state.cam_comp = cam
    state.applied = { X = offset_data.X, Y = offset_data.Y, Z = offset_data.Z }
    state.current_mode = mode

    local ok, err = add_local_offset(cam, state.applied)
    if ok then
        state.enabled = true
        return true, mode
    end
    return false, err
end

-- Обработчики кнопок
local function toggle_helmet()
    ExecuteInGameThread(function()
        if state.enabled and state.current_mode == "helmet" then
            disable_offset()
            log("Camera: Default")
        else
            local ok, res = enable_offset("helmet")
            if ok then log("Camera: Helmet Mode (Down)") else log("Error: " .. tostring(res)) end
        end
    end)
end

local function toggle_high()
    ExecuteInGameThread(function()
        if state.enabled and state.current_mode == "high" then
            disable_offset()
            log("Camera: Default")
        else
            local ok, res = enable_offset("high")
            if ok then log("Camera: High Mode (Up)") else log("Error: " .. tostring(res)) end
        end
    end)
end

-- Бинды
RegisterKeyBind(Key.F3, toggle_helmet)
RegisterKeyBind(Key.F4, toggle_high)

log("Loaded. F3: Helmet Cam, F4: High Cam")




