//=============================================================================
// ShadowEffect.fx
// 阴影渲染 Shader - 支持渐变透明度、高斯模糊和动态斜切
// 目标: MonoGame OpenGL (Shader Model 3.0)
//=============================================================================

//-----------------------------------------------------------------------------
// 参数
//-----------------------------------------------------------------------------

// 源纹理 (SpriteBatch 自动绑定到 register s0)
sampler2D SourceTexture : register(s0);

// 纹理像素尺寸 (用于模糊采样偏移计算)
float2 TexelSize;

// 视口尺寸 (用于斜切计算)
float2 ViewportSize = float2(1920, 1080);

// 渐变参数
float GradientStart = 1.0;  // 顶部透明度 (0-1)
float GradientEnd = 0.0;    // 底部透明度 (0-1)

// 模糊强度
float BlurAmount = 1.0;

// 阴影颜色 (RGBA)
float4 ShadowColor = float4(0, 0, 0, 0.2);

// 斜切参数
// ShearX: 水平斜切量（像素），正值向右，负值向左
// 表示阴影顶端（远离物体）相对于底端（靠近物体）的水平偏移
float ShearX = 0.0;

// 阴影长度缩放 (1.0 = 原始长度，用于时间变化)
float ShadowLengthScale = 1.0;

// SpriteBatch 变换矩阵 (MonoGame 自动设置)
float4x4 MatrixTransform;

//-----------------------------------------------------------------------------
// 顶点结构
//-----------------------------------------------------------------------------

struct VSInput
{
    float4 Position : POSITION0;
    float4 Color : COLOR0;
    float2 TexCoord : TEXCOORD0;
};

struct VSOutput
{
    float4 Position : SV_POSITION;
    float4 Color : COLOR0;
    float2 TexCoord : TEXCOORD0;
};

//-----------------------------------------------------------------------------
// 顶点着色器 (SpriteBatch 兼容)
//-----------------------------------------------------------------------------

// 基础顶点着色器 - 直通（用于模糊等后处理）
VSOutput SpriteVertexShader(VSInput input)
{
    VSOutput output;
    output.Position = mul(input.Position, MatrixTransform);
    output.Color = input.Color;
    output.TexCoord = input.TexCoord;
    return output;
}

//-----------------------------------------------------------------------------
// 带斜切的顶点着色器
// 实现阴影角度变化：阴影底部（靠近物体）固定，顶部（远离物体）水平偏移
//
// 关键理解：
// - SpriteBatch 使用 FlipVertically 绘制阴影
// - FlipVertically 后，TexCoord.Y=1 对应原图底部，显示在屏幕上方（阴影远端）
// - TexCoord.Y=0 对应原图顶部，显示在屏幕下方（阴影近端，靠近物体）
// - 我们需要固定 Y=0（靠近物体），偏移 Y=1（远离物体）
//-----------------------------------------------------------------------------

VSOutput ShearVertexShader(VSInput input)
{
    VSOutput output;
    output.Color = input.Color;
    output.TexCoord = input.TexCoord;
    
    float4 pos = input.Position;
    
    // 从 Color.RG 获取源矩形的 Y 边界（归一化到 0-1）
    // Color.R = srcRect.Top / texHeight
    // Color.G = srcRect.Bottom / texHeight
    // Color.B = 精灵渲染高度 / 512（归一化）
    float srcTop = input.Color.r;
    float srcBottom = input.Color.g;
    float normalizedHeight = input.Color.b;
    float range = srcBottom - srcTop;
    
    // 计算当前顶点在精灵内的相对位置 (0 = 顶部, 1 = 底部)
    float normalizedV = 0.5;
    if (range > 0.001)
    {
        normalizedV = saturate((input.TexCoord.y - srcTop) / range);
    }
    
    // 恢复精灵的实际渲染高度（像素）
    float spriteHeight = normalizedHeight * 512.0;
    
    // FlipVertically 后：
    // - normalizedV = 0 是原图顶部 -> 显示在屏幕下方（阴影远端）-> 最大偏移
    // - normalizedV = 1 是原图底部 -> 显示在屏幕上方（靠近物体）-> 固定不动
    float shearFactor = 1.0 - normalizedV;
    
    // ShearX 现在表示"角度因子"（tan(angle) * 100）
    // 实际像素偏移 = (ShearX / 100) * spriteHeight * shearFactor
    // 这样所有物体都有相同的倾斜角度
    float pixelOffset = (ShearX / 100.0) * spriteHeight * shearFactor;
    
    pos.x += pixelOffset;
    
    // 应用 SpriteBatch 的变换矩阵
    output.Position = mul(pos, MatrixTransform);
    
    return output;
}

//-----------------------------------------------------------------------------
// Technique 1: 像素级渐变阴影
// 使用 Color.RG 通道传递源矩形边界，计算精灵内的相对 V 坐标
// Color.R = srcRect.Top / texHeight
// Color.G = srcRect.Bottom / texHeight
// 这样可以实现真正的像素级渐变，不受 spritesheet 位置影响
//-----------------------------------------------------------------------------

float4 GradientShadowPS(VSOutput input) : COLOR
{
    // 采样原始纹理
    float4 texColor = tex2D(SourceTexture, input.TexCoord);
    
    // 如果原始像素透明，跳过
    if (texColor.a < 0.01)
        discard;
    
    // 从 Color.RG 获取源矩形的 Y 边界（归一化到 0-1）
    float srcTop = input.Color.r;
    float srcBottom = input.Color.g;
    
    // 计算当前像素在精灵内的相对 V 坐标 (0 = 顶部, 1 = 底部)
    float normalizedV = 0.5; // 默认值
    float range = srcBottom - srcTop;
    if (range > 0.001)
    {
        normalizedV = saturate((input.TexCoord.y - srcTop) / range);
    }
    
    // 计算渐变因子
    // normalizedV = 0 是原图顶部 -> FlipVertically 后是阴影底部（远离物体）-> 应该淡
    // normalizedV = 1 是原图底部 -> FlipVertically 后是阴影顶部（靠近物体）-> 应该深
    
    // 使用较平缓的非线性渐变：pow(1-x, 1.5) 让渐变更均匀
    // 反转 normalizedV 使得 v=1(靠近物体) 对应 curvedV=0(最深)
    float curvedV = pow(1.0 - normalizedV, 1.5);
    // curvedV=0 (靠近物体) -> GradientStart (深)
    // curvedV=1 (远离物体) -> GradientEnd (淡)
    float gradientFactor = lerp(GradientStart, GradientEnd, curvedV);
    
    // 输出阴影颜色
    float4 result = ShadowColor;
    result.a *= gradientFactor * texColor.a;
    
    return result;
}

technique GradientShadow
{
    pass P0
    {
        VertexShader = compile vs_3_0 ShearVertexShader();
        PixelShader = compile ps_3_0 GradientShadowPS();
    }
}

//-----------------------------------------------------------------------------
// Technique 2: 水平高斯模糊
//-----------------------------------------------------------------------------

// 9-tap 高斯权重 (sigma ≈ 2.0)
static const float Weights[5] = { 0.227027, 0.1945946, 0.1216216, 0.054054, 0.016216 };

float4 HorizontalBlurPS(VSOutput input) : COLOR
{
    float4 color = float4(0, 0, 0, 0);
    
    // 中心像素
    color += tex2D(SourceTexture, input.TexCoord) * Weights[0];
    
    // 两侧采样
    for (int i = 1; i < 5; i++)
    {
        float2 offset = float2(TexelSize.x * i * BlurAmount, 0);
        color += tex2D(SourceTexture, input.TexCoord + offset) * Weights[i];
        color += tex2D(SourceTexture, input.TexCoord - offset) * Weights[i];
    }
    
    return color;
}

technique HorizontalBlur
{
    pass P0
    {
        PixelShader = compile ps_3_0 HorizontalBlurPS();
    }
}

//-----------------------------------------------------------------------------
// Technique 3: 垂直高斯模糊
//-----------------------------------------------------------------------------

float4 VerticalBlurPS(VSOutput input) : COLOR
{
    float4 color = float4(0, 0, 0, 0);
    
    // 中心像素
    color += tex2D(SourceTexture, input.TexCoord) * Weights[0];
    
    // 两侧采样
    for (int i = 1; i < 5; i++)
    {
        float2 offset = float2(0, TexelSize.y * i * BlurAmount);
        color += tex2D(SourceTexture, input.TexCoord + offset) * Weights[i];
        color += tex2D(SourceTexture, input.TexCoord - offset) * Weights[i];
    }
    
    return color;
}

technique VerticalBlur
{
    pass P0
    {
        PixelShader = compile ps_3_0 VerticalBlurPS();
    }
}

//-----------------------------------------------------------------------------
// Technique 4: 简单阴影 (无渐变，仅颜色替换)
//-----------------------------------------------------------------------------

float4 SimpleShadowPS(VSOutput input) : COLOR
{
    float4 texColor = tex2D(SourceTexture, input.TexCoord);
    
    if (texColor.a < 0.01)
        discard;
    
    float4 result = ShadowColor;
    result.a *= texColor.a * input.Color.a;
    
    return result;
}

technique SimpleShadow
{
    pass P0
    {
        PixelShader = compile ps_3_0 SimpleShadowPS();
    }
}

//-----------------------------------------------------------------------------
// Technique 5: 组合效果 (渐变 + 单 Pass 模糊近似)
// 使用 5x5 采样实现轻量级模糊
//-----------------------------------------------------------------------------

float4 GradientBlurShadowPS(VSOutput input) : COLOR
{
    float4 color = float4(0, 0, 0, 0);
    float totalWeight = 0;
    
    // 5x5 采样模式 (简化的高斯近似)
    for (int y = -2; y <= 2; y++)
    {
        for (int x = -2; x <= 2; x++)
        {
            float2 offset = float2(x, y) * TexelSize * BlurAmount;
            float4 sample = tex2D(SourceTexture, input.TexCoord + offset);
            
            // 距离权重
            float weight = 1.0 / (1.0 + abs(x) + abs(y));
            color += sample * weight;
            totalWeight += weight;
        }
    }
    
    color /= totalWeight;
    
    // 如果采样区域全透明，跳过
    if (color.a < 0.01)
        discard;
    
    // 应用渐变
    float gradientFactor = lerp(GradientStart, GradientEnd, input.TexCoord.y);
    
    // 输出阴影
    float4 result = ShadowColor;
    result.a *= gradientFactor * color.a * input.Color.a;
    
    return result;
}

technique GradientBlurShadow
{
    pass P0
    {
        PixelShader = compile ps_3_0 GradientBlurShadowPS();
    }
}

//-----------------------------------------------------------------------------
// Technique 6: 屏幕空间渐变后处理
// 在整个 RenderTarget 上应用从上到下的渐变
// 用于解决不同精灵因 spritesheet 位置不同导致渐变不一致的问题
//-----------------------------------------------------------------------------

float4 ScreenGradientPS(VSOutput input) : COLOR
{
    float4 texColor = tex2D(SourceTexture, input.TexCoord);
    
    // 如果透明，跳过
    if (texColor.a < 0.01)
        discard;
    
    // 基于屏幕 Y 坐标计算渐变
    // Y = 0 是屏幕顶部（阴影底部，远离物体）= GradientEnd
    // Y = 1 是屏幕底部（阴影顶部，靠近物体）= GradientStart
    // 注意：阴影是翻转的，所以物体在下方，阴影向上延伸
    float gradientFactor = lerp(GradientEnd, GradientStart, input.TexCoord.y);
    
    float4 result = texColor;
    result.a *= gradientFactor;
    
    return result;
}

technique ScreenGradient
{
    pass P0
    {
        PixelShader = compile ps_3_0 ScreenGradientPS();
    }
}
